/**
 * @file device.c
 * @author TheGreenBow (support@thegreenbow.com)
 * @brief 
 * @version 0.1
 * @date 2020-06-15
 * 
 * @copyright Copyright (c) 2020
 * 
 */

#include "device.h"

#include <linux/module.h>
#include <linux/kernel.h>

#include <linux/netdevice.h>
#include <linux/skbuff.h>


#include "send.h"


#define tgb79f1739fd66f7db1ef3588ef6a20ff4b1a29a27fe616a0c54e6c9ec9694548d2	  0xFFFE
#define tgbcfaef0ad5ebb251155c52bbaf1e4810ba30f91f06032208c75a815269aa903fe NETIF_F_HW_CSUM | NETIF_F_RXCSUM | NETIF_F_SG | NETIF_F_GSO | NETIF_F_HIGHDMA

/**
 * @brief function called when interface is up
 * 
 * @param dev net_device
 * @return int 
 */
int tgbtun_open(struct net_device *dev) {
	return 0;
}

/**
 * @brief function called when interface is released
 * 
 * @param dev net_device
 * @return int 
 */
int tgbtun_release(struct net_device *dev) {
	netif_stop_queue(dev);
	return 0;
}

/**
 * @brief function called when packet is send
 * 
 * @param skb sending packet
 * @param dev net_device
 * @return int 
 */
int tgbtun_xmit(struct sk_buff *skb, struct net_device *dev) {
	dev->stats.tx_packets++;
	dev->stats.tx_bytes += skb->len;
	send_packet(skb);
	return NETDEV_TX_OK;
}

/**
 * @brief function called when interface type is created
 * 
 * @param dev net_device
 * @return int 
 */
int tgbtun_init(struct net_device *dev) {
  	return 0;
}

/**
 * @brief net_device_ops
 * 
 */
const struct net_device_ops tgbtun_netdev_ops = {
	.ndo_init = tgbtun_init,
    .ndo_open = tgbtun_open,
    .ndo_stop = tgbtun_release,
    .ndo_start_xmit = tgbtun_xmit,
};


static void virtual_setup(struct net_device *dev){
	dev->netdev_ops = &tgbtun_netdev_ops;
	dev->hard_header_len = 0; //Maximum hardware header length.
	dev->addr_len = 0; //Hardware address length
	dev->type = tgb79f1739fd66f7db1ef3588ef6a20ff4b1a29a27fe616a0c54e6c9ec9694548d2; //Interface hardware type
	dev->flags = IFF_POINTOPOINT | IFF_NOARP; //Declare POINTOPOINT and NOARP interface
	dev->priv_flags |= IFF_NO_QUEUE; //Like flags but invisible to userspace
	dev->features |= tgbcfaef0ad5ebb251155c52bbaf1e4810ba30f91f06032208c75a815269aa903fe; //Currently active device features
	dev->hw_features |= tgbcfaef0ad5ebb251155c52bbaf1e4810ba30f91f06032208c75a815269aa903fe; //User-changeable features
	dev->hw_enc_features |= tgbcfaef0ad5ebb251155c52bbaf1e4810ba30f91f06032208c75a815269aa903fe; //Mask of features inherited by encapsulating devices
	dev->mtu = 1300; //Interface MTU value
	dev->max_mtu = round_down(INT_MAX, 16); //Interface Maximum MTU value

	// We need to keep the dst around in case of icmp replies.
	netif_keep_dst(dev);
}


struct net_device *tgbtun;
bool device_up = false;

static void unload_all_inteface(void) {
	if(device_up){
		unregister_netdev(tgbtun);
		device_up = false;
	}
}

/**
 * @brief load module
 * 
 * @return int 
 */
int device_load(void) {
 	int result;

	if(!device_up) {
		tgbtun = alloc_netdev (0, "tgbtun%d", NET_NAME_UNKNOWN, virtual_setup);

		if((result = register_netdev(tgbtun))) {
			pr_err("tgbtun: Error %d initalizing card ...", result);
  			return result;
 		}
		device_up = true;
	}
	
 	return 0;
}

/**
 * @brief unload module
 * 
 */
void device_unload(void) {
	unload_all_inteface();
}
