
#include "ioctl.h"

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/ioctl.h>
#include <linux/uaccess.h>
#include <linux/cdev.h>

#include <linux/netdevice.h>
#include <linux/inetdevice.h>
#include <linux/inet.h>
#include "private_api.h"

#define tgb8f7ec51f94e33d00048ebd548b649bba6f811fb7a188a8983baef96588d59948 "tgbtun"

#define tgb9b98ed9cadac5aa42c5fc85d22dc89f0ecfec023003b680fe6399fdc777370d2 _IOW('a', 'a', void *)  // get driver status
#define tgb7fec5d2cb9297991aa9dbda61ff97904629ffa5321c7369a340952a83507b798 _IOR('a', 'b', void *)   // send command to driver

typedef struct
{
    atomic_t available;
    struct semaphore sem;
    struct cdev cdev;
} ioctl_d_interface_dev;

int ioctl_d_interface_major = 0;
int ioctl_d_interface_minor = 0;

char *ioctl_d_interface_name = tgb8f7ec51f94e33d00048ebd548b649bba6f811fb7a188a8983baef96588d59948;

ioctl_d_interface_dev ioctl_d_interface;

int ioctl_open(struct inode *inode, struct file *filp);
int ioctl_release(struct inode *inode, struct file *filp);
long ioctl_unlocked_ioctl(struct file *filp, unsigned int cmd, unsigned long arg);

struct file_operations ioctl_d_interface_fops = {.owner = THIS_MODULE, .read = NULL, .write = NULL, .open = ioctl_open, .unlocked_ioctl = ioctl_unlocked_ioctl, .release = ioctl_release};

// Private API
static int ioctl_dev_init(ioctl_d_interface_dev *ioctl_d_interface)
{
    int result = 0;
    memset(ioctl_d_interface, 0, sizeof(ioctl_d_interface_dev));
    atomic_set(&ioctl_d_interface->available, 1);
    sema_init(&ioctl_d_interface->sem, 1);

    return result;
}

static int ioctl_setup_cdev(ioctl_d_interface_dev *ioctl_d_interface)
{
    int error = 0;
    dev_t devno = MKDEV(ioctl_d_interface_major, ioctl_d_interface_minor);

    cdev_init(&ioctl_d_interface->cdev, &ioctl_d_interface_fops);
    ioctl_d_interface->cdev.owner = THIS_MODULE;
    ioctl_d_interface->cdev.ops = &ioctl_d_interface_fops;
    error = cdev_add(&ioctl_d_interface->cdev, devno, 1);

    return error;
}

// Public API
int ioctl_open(struct inode *inode, struct file *filp)
{
    ioctl_d_interface_dev *ioctl_d_interface;

    ioctl_d_interface = container_of(inode->i_cdev, ioctl_d_interface_dev, cdev);
    filp->private_data = ioctl_d_interface;

    if (!atomic_dec_and_test(&ioctl_d_interface->available))
    {
        atomic_inc(&ioctl_d_interface->available);
        pr_alert("open ioctl_d_interface : the device has been opened by some other device, unable to open lock\n");
        return -EBUSY;  // already open
    }

    return 0;
}

int ioctl_release(struct inode *inode, struct file *filp)
{
    ioctl_d_interface_dev *ioctl_d_interface = filp->private_data;
    atomic_inc(&ioctl_d_interface->available);  // release the device
    return 0;
}

/**
 * @brief ioctl handle, catch command
 *
 * @param filp
 * @param cmd command receive
 * @param arg command arg
 * @return long
 */
long ioctl_unlocked_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    void *in, *out;
    in = kmalloc(DRIVER_CMD_MSG_SIZE, GFP_ATOMIC);
    if (unlikely(!in))
        return -EFAULT;
    out = kmalloc(DRIVER_CMD_MSG_SIZE, GFP_ATOMIC);
    if (unlikely(!out))
        return -EFAULT;

    switch (cmd)
    {
        case tgb9b98ed9cadac5aa42c5fc85d22dc89f0ecfec023003b680fe6399fdc777370d2: {
            p_reset_device();
            p_reset_ipsec();
            p_send_start_msg_to_deamon(out);
            if (copy_to_user((int32_t *)arg, out, DRIVER_CMD_MSG_SIZE))
            {
                pr_err("tgbtun: copy from user space failed\n");
                return -EFAULT;
            }
            break;
        }

        case tgb7fec5d2cb9297991aa9dbda61ff97904629ffa5321c7369a340952a83507b798: {
            if (copy_from_user(in, (int32_t *)arg, DRIVER_CMD_MSG_SIZE))
            {
                pr_err("tgbtun: copy from user space failed\n");
                return -EFAULT;
            }
            p_income_msg_from_daemon(in);
            break;
        }
        default:
            pr_err("tgbtun: iotcl bad command %d\n", cmd);
    }
    kfree(in);
    kfree(out);
    return 0;
}

/**
 * @brief Load ioctl, alloc nod and device
 *
 * @return int
 */
int ioctl_load(void)
{
    dev_t devno = 0;
    int result = 0;

    ioctl_dev_init(&ioctl_d_interface);

    // register char device
    // we will get the major number dynamically this is recommended see
    // book : ldd3
    //
    result = alloc_chrdev_region(&devno, ioctl_d_interface_minor, 1, ioctl_d_interface_name);
    ioctl_d_interface_major = MAJOR(devno);
    if (result < 0)
    {
        printk(KERN_WARNING "tgbtun: ioctl_d_interface: can't get major number %d\n", ioctl_d_interface_major);
        goto fail;
    }

    result = ioctl_setup_cdev(&ioctl_d_interface);
    if (result < 0)
    {
        printk(KERN_WARNING "tgbtun: ioctl_d_interface: error %d adding ioctl_d_interface", result);
        goto fail;
    }
    return 0;

fail:
    ioctl_unload();
    return result;
}

/**
 * @brief Unload ioctl
 *
 */
void ioctl_unload(void)
{
    dev_t devno = MKDEV(ioctl_d_interface_major, ioctl_d_interface_minor);

    cdev_del(&ioctl_d_interface.cdev);
    unregister_chrdev_region(devno, 1);
}
