/**
 * @file netfilter.c
 * @author TheGreenBow (support@thegreenbow.com)
 * @copyright Copyright (c) 2025
 */

#include "netfilter.h"

#include <linux/netfilter_ipv4.h>
#include <linux/skbuff.h>
#include <linux/ip.h>
#include <linux/udp.h>

#include "frame_utils.h"
#include "receive.h"
#include "private_api.h"

/**
 * @brief Netfilter hook, this function will catch all inbound packet
 *
 * @param priv
 * @param skb
 * @param state
 * @return unsigned int
 */
unsigned int hook_func_in(void *priv, struct sk_buff *skb, const struct nf_hook_state *state)
{
    struct sk_buff *my_skb, *trailer;

    if (unlikely(!skb))
        return NF_DROP;  // if skb null
    if (skb->protocol != htons(ETH_P_IP))
        return NF_ACCEPT;  // if is not IP packet

    skb_data_align(skb, 0, &trailer);  // for fragmentation

    switch (ip_hdr(skb)->protocol)
    {
        case IPPROTO_ESP: {
            goto esp_packet;
            break;
        }
        case IPPROTO_UDP: {
            u_int16_t nat_port = 4500;
            p_get_natport(&nat_port);
            if (udp_hdr(skb)->source == htons(nat_port))
            {
                goto esp_packet;
            }
            else
            {
                return NF_ACCEPT;
            }
            break;
        }
        default: {
            return NF_ACCEPT;
        }
    }

esp_packet:
    my_skb = skb_clone(skb, GFP_ATOMIC);  // allocate a network buffer
    if (unlikely(tgb7a0764f6e74752a830e190e54b5e0be578e0e17a05726f034daa75202a7a19cd(my_skb) == false))
    {
        return NF_ACCEPT;
    }
    kfree_skb(my_skb);

    return NF_DROP;
}

static struct nf_hook_ops in_packet = {
    .hook = hook_func_in,
    .hooknum = NF_INET_LOCAL_IN,  // (capture in packet)
    .pf = NFPROTO_IPV4,           // IPV4 packets
    .priority = NF_IP_PRI_LAST,
};

int netfilter_load(void)
{
    nf_register_net_hook(&init_net, &in_packet);  // register hook
    return 0;
}

void netfilter_unload(void) { nf_unregister_net_hook(&init_net, &in_packet); }