
#include "send.h"

#include <linux/skbuff.h>
#include <linux/ip.h>
#include <linux/inet.h>
#include <linux/inetdevice.h>

#include "frame_utils.h"
#include "private_api.h"

#include "debug.h"

extern unsigned int ip_phys;
extern uint8_t mac_dst[ETH_ALEN];
struct net_device *physcal_dev;

static DEFINE_SPINLOCK(devices_lock);

bool send_packet(struct sk_buff *skb)
{
    struct ethhdr *eth;
    struct sk_buff *trailer;
    unsigned int status;

    skb_data_align(skb, 0, &trailer);
    eth = eth_hdr(skb);

    eth = (struct ethhdr *)skb_push(skb, sizeof(struct ethhdr));  // add data to the start of a buffer
    if (ip_hdr(skb)->version == 4)
    {
        eth->h_proto = be16_to_cpu(ETH_P_IP);
    }
    else if (ip_hdr(skb)->version == 6)
    {
        eth->h_proto = be16_to_cpu(ETH_P_IPV6);
    }
    else
    {
        pr_warn("tgbtun: Not an ipv4 or ipv6 packet\n");
        dev_kfree_skb(skb);
        return true;
    }

    status = p_api_frame_encrypt(eth->h_dest, skb->len);

    dev_kfree_skb(skb);
    if (unlikely(status == 0))
        return false;
    return true;
}

bool tgb632fda97a51b0f9507a661fe9bbd846f692e13c540fa73d485d7c19285866eef(uint8_t *frame, const uint32_t size)
{
    struct sk_buff *skb, *trailer;

    if (unlikely(!frame) || !size)
        return false;

    skb = alloc_skb(size + ETH_HLEN + 2, GFP_ATOMIC);
    if (unlikely(!skb))
        return false;

    memcpy(skb_put(skb, size), frame, size);
    kfree(frame);  // free buffer get from ipseclib

    skb_reset_mac_header(skb);
    skb->dev = physcal_dev;
    skb->pkt_type = PACKET_OUTGOING;

    skb_data_align(skb, 1, &trailer);

    if (unlikely(!physcal_dev) || unlikely(!(physcal_dev->dev_addr)))
    {
        pr_warn("tgbtun: no physical device for sending packets\n");
        kfree_skb(skb);
        return false;
    }

    memcpy(eth_hdr(skb)->h_source, physcal_dev->dev_addr, ETH_ALEN);
    memcpy(eth_hdr(skb)->h_dest, mac_dst, ETH_ALEN);

    dev_queue_xmit(skb);

    return true;
}

bool send_init(void)
{
    struct net_device *dev;
    struct in_device *in_dev;
    struct in_ifaddr *if_info;
    bool found = false;
    unsigned long flags = 0;

    spin_lock_irqsave(&devices_lock, flags);

    dev = first_net_device(&init_net);
    while (dev && found == false)
    {

        in_dev = (struct in_device *)dev->ip_ptr;
        if_info = in_dev->ifa_list;
        for (; if_info; if_info = if_info->ifa_next)
        {
            if (if_info->ifa_address == ip_phys)
            {
                pr_debug("tgbtun: found device [%s]\n", dev->name);
                found = true;
                break;
            }
        }
        if (!found)
            dev = next_net_device(dev);
    }

    spin_unlock_irqrestore(&devices_lock, flags);

    if (likely(found))
    {
        physcal_dev = dev;
    }
    else
    {
        pr_err("tgbtun: Physical device not found for ip : %pI4", &ip_phys);
        return false;
    }
    return true;
}
